<?php

namespace App\Helpers;

use App\Models\Transaction;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class PayPalHelper
{
    /**
     * Generate a PayPal payment URL
     *
     * @param Transaction $transaction
     * @param Package $package
     * @return string
     */
    public static function generatePaymentUrl($transaction, $package)
    {
        // Get PayPal configuration
        $paypalEmail = config_value('store_paypal_email', '');
        $paypalCurrency = config_value('store_currency', 'GBP');
        $returnUrl = route('payment.success', $transaction->uuid);
        $cancelUrl = route('payment.cancel', $transaction->uuid);
        $notifyUrl = route('payment.ipn');
        
        // Generate PayPal payment URL
        $paypalUrl = boolval(config_value('store_sandbox')) 
        ? 'https://www.sandbox.paypal.com/cgi-bin/webscr' 
        : 'https://www.paypal.com/cgi-bin/webscr';
        
        $paypalUrl .= '?cmd=_xclick';
        $paypalUrl .= '&business=' . urlencode($paypalEmail);
        $paypalUrl .= '&item_name=' . urlencode($package->name . ' - EdgeRP');
        $paypalUrl .= '&item_number=' . urlencode($transaction->uuid);
        $paypalUrl .= '&amount=' . urlencode($package->price);
        $paypalUrl .= '&currency_code=' . urlencode($paypalCurrency);
        $paypalUrl .= '&return=' . urlencode($returnUrl);
        $paypalUrl .= '&cancel_return=' . urlencode($cancelUrl);
        $paypalUrl .= '&notify_url=' . urlencode($notifyUrl);
        $paypalUrl .= '&custom=' . urlencode(json_encode([
            'user_id' => Auth::id(),
            'package_id' => $package->id,
        ]));
        
        return $paypalUrl;
    }
    
    /**
     * Process IPN verification
     *
     * @param array $data
     * @return bool
     */
    public static function verifyIPN($data)
    {
        try {
            $ipn = app('paypal.ipn');
            
            // Use the sandbox endpoint during testing
            if (config('app.env') === 'production') {
                $ipn->useProd();
            } else {
                $ipn->useSandbox();
            }
            
            return $ipn->verifyIPN();
        } catch (\Exception $e) {
            Log::error('PayPal IPN Error: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Process the package benefits for the user
     *
     * @param Transaction $transaction
     * @return void
     */
    public static function processPackageBenefits($transaction)
    {
    }
    
    /**
     * Extract money amount from package features
     * 
     * @param Package $package
     * @return int
     */
    private static function extractMoneyAmount($package)
    {
        foreach ($package->features as $feature) {
            if (preg_match('/\$([0-9,]+)/', $feature, $matches)) {
                return (int) str_replace(',', '', $matches[1]);
            }
        }
        
        return 0;
    }
    
    /**
     * Extract job type from package name
     * 
     * @param Package $package
     * @return string
     */
    private static function extractJobType($package)
    {
        return '';
    }
}